﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Jint.Runtime.VM {
    class Frame {
        AbstractBox[] m_registers;
        RuntimeContext m_runtime;

        public Frame(int size, RuntimeContext runtime) {
            if (runtime == null)
                throw new ArgumentNullException("runtime");
            if (size < 0)
                throw new ArgumentOutOfRangeException("size");
            m_runtime = runtime;
            m_registers = new AbstractBox[size];
        }

        public AbstractBox this[int index] {
            get {
                return m_registers[index];
            }
            set {
                m_registers[index] = value;
            }
        }

        /// <summary>
        /// Extracts value stored in the registry specified by the index.
        /// </summary>
        /// <remarks>This method doesn't do any cast, if the specified type isn't the same as the type of the stored value a type cast exception will occur.</remarks>
        /// <typeparam name="T">The type of the value stored in the registry.</typeparam>
        /// <param name="index">The index of the registry.</param>
        /// <returns>The value stored in the registry.</returns>
        public T GetValue<T>(int index) {
            return ((Box<T>)m_registers[index]).holdingValue;
        }

        /// <summary>
        /// Stores a new value in the register specified by the index.
        /// </summary>
        /// <remarks>
        /// If the previous value has the same type as the value being stored in the registry,
        /// the new value will replace the old one, otherwise the registry will be reallocated to
        /// store the new value.
        /// </remarks>
        /// <typeparam name="T">The type of the value being stored</typeparam>
        /// <param name="index">The index of the registry where the value will be stored</param>
        /// <param name="value">The value to be stored in the registry</param>
        public void SetValue<T>(int index, T value) {
            var reg = m_registers[index] as Box<T>;
            if (reg == null || reg.holdingType != typeof(T))
                m_registers[index] = m_runtime.BoxValue(value);
            else
                reg.holdingValue = value;
        }
    }
}
